<?php
/**
 * Admin settings for Floating Promo Banner plugin
 */

class FloatingPromoAdmin {

	public function __construct() {
		add_action('admin_menu', array($this, 'add_admin_menu'));
		add_action('admin_init', array($this, 'settings_init'));
		add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
	}

	public function enqueue_admin_scripts($hook) {
        // Only load on our settings page
		if ($hook !== 'settings_page_floating-promo') {
			return;
		}
		
        // Enqueue media uploader
		wp_enqueue_media();
		
        // Enqueue admin JavaScript
		wp_enqueue_script(
			'floating-promo-admin-script',
			FLOATING_PROMO_PLUGIN_URL . 'js/admin.js',
			array('jquery'),
			FLOATING_PROMO_VERSION,
			true
		);
		
        // Add custom CSS for admin
		wp_add_inline_style('wp-admin', '
			.floating-promo-image-upload .button {
				margin-right: 10px;
				margin-bottom: 10px;
			}
            #banner_image_preview img {
			max-width: 100%;
			height: auto;
			border-radius: 5px;
			border: 1px solid #ddd;
			margin-bottom: 10px;
		}
		.recommended-size {
			display: block;
			margin-top: 5px;
			color: #666;
			font-style: italic;
		}
		');
	}

	public function add_admin_menu() {
		add_options_page(
			__('Floating Promo Settings', 'floating-promo'),
			__('Floating Promo', 'floating-promo'),
			'manage_options',
			'floating-promo',
			array($this, 'options_page')
		);
	}

	public function settings_init() {
		register_setting('floating_promo', 'floating_promo_settings');

		add_settings_section(
			'floating_promo_section',
			__('Pengaturan Banner Promo', 'floating-promo'),
			array($this, 'settings_section_callback'),
			'floating_promo'
		);

		add_settings_field(
			'enable_banner',
			__('Aktifkan Banner', 'floating-promo'),
			array($this, 'enable_banner_render'),
			'floating_promo',
			'floating_promo_section'
		);

		add_settings_field(
			'banner_position',
			__('Posisi Default Banner', 'floating-promo'),
			array($this, 'banner_position_render'),
			'floating_promo',
			'floating_promo_section'
		);

		add_settings_field(
			'banner_image',
			__('Gambar Banner', 'floating-promo'),
			array($this, 'banner_image_render'),
			'floating_promo',
			'floating_promo_section'
		);

		add_settings_field(
			'banner_content',
			__('Konten Teks Banner', 'floating-promo'),
			array($this, 'banner_content_render'),
			'floating_promo',
			'floating_promo_section'
		);

		add_settings_field(
			'banner_url',
			__('URL Tujuan', 'floating-promo'),
			array($this, 'banner_url_render'),
			'floating_promo',
			'floating_promo_section'
		);

		add_settings_field(
			'excluded_pages',
			__('Halaman yang Dikecualikan', 'floating-promo'),
			array($this, 'excluded_pages_render'),
			'floating_promo',
			'floating_promo_section'
		);
	}

	public function enable_banner_render() {
		$options = get_option('floating_promo_settings');
		?>
		<input type="checkbox" name="floating_promo_settings[enable_banner]" <?php checked(isset($options['enable_banner']) ? $options['enable_banner'] : false, 1); ?> value="1">
		<label for="floating_promo_settings[enable_banner]"><?php _e('Tampilkan banner promo', 'floating-promo'); ?></label>
		<?php
	}

	public function banner_position_render() {
		$options = get_option('floating_promo_settings');
		$position = isset($options['banner_position']) ? $options['banner_position'] : 'center';
		?>
		<select name="floating_promo_settings[banner_position]">
			<option value="left" <?php selected($position, 'left'); ?>><?php _e('Kiri', 'floating-promo'); ?></option>
			<option value="center" <?php selected($position, 'center'); ?>><?php _e('Tengah', 'floating-promo'); ?></option>
			<option value="right" <?php selected($position, 'right'); ?>><?php _e('Kanan', 'floating-promo'); ?></option>
		</select>
		<p class="description"><?php _e('Posisi default banner ketika pertama kali ditampilkan.', 'floating-promo'); ?></p>
		<?php
	}

	public function banner_image_render() {
		$options = get_option('floating_promo_settings');
		$image_url = isset($options['banner_image']) ? $options['banner_image'] : '';
		?>
		<div class="floating-promo-image-upload">
			<input type="hidden" name="floating_promo_settings[banner_image]" id="banner_image_url" value="<?php echo esc_url($image_url); ?>">
			<div id="banner_image_preview" style="margin-bottom: 15px;">
				<?php if (!empty($image_url)) : ?>
					<img src="<?php echo esc_url($image_url); ?>" style="max-width: 300px; height: auto;">
					<div class="recommended-size"><?php _e('Rekomendasi ukuran: 700x800px', 'floating-promo'); ?></div>
				<?php endif; ?>
			</div>
			<button type="button" class="button button-primary" id="upload_banner_image"><?php _e('Pilih Gambar', 'floating-promo'); ?></button>
			<button type="button" class="button" id="remove_banner_image" style="<?php echo empty($image_url) ? 'display:none;' : ''; ?>"><?php _e('Hapus Gambar', 'floating-promo'); ?></button>
			<p class="description"><?php _e('Unggah gambar untuk banner promosi. Ukuran disarankan: 700x800px.', 'floating-promo'); ?></p>
		</div>
		<?php
	}

	public function banner_content_render() {
		$options = get_option('floating_promo_settings');
		$content = isset($options['banner_content']) ? $options['banner_content'] : '';
		wp_editor($content, 'floating_promo_banner_content', array(
			'textarea_name' => 'floating_promo_settings[banner_content]',
			'textarea_rows' => 5,
			'media_buttons' => false,
			'tinymce' => array(
				'toolbar1' => 'bold,italic,underline,link,undo,redo'
			)
		));
		?>
		<p class="description"><?php _e('Konten teks banner promosi yang akan ditampilkan. HTML diperbolehkan.', 'floating-promo'); ?></p>
		<?php
	}

	public function banner_url_render() {
		$options = get_option('floating_promo_settings');
		$url = isset($options['banner_url']) ? $options['banner_url'] : '';
		?>
		<input type="url" name="floating_promo_settings[banner_url]" value="<?php echo esc_url($url); ?>" class="regular-text">
		<p class="description"><?php _e('URL tujuan ketika banner diklik.', 'floating-promo'); ?></p>
		<?php
	}

	public function excluded_pages_render() {
		$options = get_option('floating_promo_settings');
		$excluded = isset($options['excluded_pages']) ? $options['excluded_pages'] : array();
		
		$pages = get_pages();
		?>
		<select name="floating_promo_settings[excluded_pages][]" multiple="multiple" style="width: 100%; min-height: 150px;">
			<?php foreach ($pages as $page) : ?>
				<option value="<?php echo $page->ID; ?>" <?php selected(in_array($page->ID, $excluded), true); ?>>
					<?php echo esc_html($page->post_title); ?>
				</option>
			<?php endforeach; ?>
		</select>
		<p class="description"><?php _e('Pilih halaman dimana banner TIDAK akan ditampilkan. Gunakan Ctrl/Cmd untuk memilih multiple.', 'floating-promo'); ?></p>
		<?php
	}

	public function settings_section_callback() {
		echo '<p>' . __('Konfigurasi banner promosi yang akan ditampilkan secara floating di website Anda.', 'floating-promo') . '</p>';
	}

	public function options_page() {
		?>
		<div class="wrap">
			<h1><?php _e('Pengaturan Floating Promo', 'floating-promo'); ?></h1>
			
			<form action="options.php" method="post">
				<?php
				settings_fields('floating_promo');
				do_settings_sections('floating_promo');
				submit_button();
				?>
			</form>
			
			<div style="margin-top: 30px; padding: 15px; background: #f5f5f5; border-left: 4px solid #0073aa;">
				<h3><?php _e('Cara Penggunaan', 'floating-promo'); ?></h3>
				<ol>
					<li><?php _e('Aktifkan banner dengan mencentang opsi "Aktifkan Banner"', 'floating-promo'); ?></li>
					<li><?php _e('Pilih posisi default banner (kiri, tengah, atau kanan)', 'floating-promo'); ?></li>
					<li><?php _e('Unggah gambar banner dengan ukuran optimal 700x800px', 'floating-promo'); ?></li>
					<li><?php _e('Isi konten teks banner promosi Anda (bisa menggunakan HTML sederhana)', 'floating-promo'); ?></li>
					<li><?php _e('Tentukan URL tujuan ketika banner diklik', 'floating-promo'); ?></li>
					<li><?php _e('Pilih halaman yang tidak ingin menampilkan banner (opsional)', 'floating-promo'); ?></li>
					<li><?php _e('Simpan perubahan', 'floating-promo'); ?></li>
				</ol>
				<p><strong><?php _e('Fitur untuk Pengunjung:', 'floating-promo'); ?></strong></p>
				<ul style="list-style-type: disc; margin-left: 20px;">
					<li><?php _e('Dapat memindahkan banner ke kiri, tengah, atau kanan layar', 'floating-promo'); ?></li>
					<li><?php _e('Dapat menutup banner (akan tersimpan selama 30 hari)', 'floating-promo'); ?></li>
					<li><?php _e('Responsif di perangkat mobile', 'floating-promo'); ?></li>
					<li><?php _e('Tampilan gambar dengan kualitas tinggi (700x800px)', 'floating-promo'); ?></li>
				</ul>
			</div>
		</div>
		<?php
	}
}