<?php
/**
 * Plugin Name: Floating Promo Banner
 * Plugin URI: https://example.com/floating-promo
 * Description: Plugin untuk menampilkan banner promosi yang dapat dipindahkan dan ditutup oleh pengguna.
 * Version: 1.1.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: floating-promo
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('FLOATING_PROMO_VERSION', '1.1.0');
define('FLOATING_PROMO_PLUGIN_URL', plugin_dir_url(__FILE__));
define('FLOATING_PROMO_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Main plugin class
class FloatingPromo {

    private static $instance = null;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
    }

    public function init() {
        // Load text domain for translations
        load_plugin_textdomain('floating-promo', false, dirname(plugin_basename(__FILE__)) . '/languages');

        // Enqueue frontend assets
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));

        // Include admin settings
        if (is_admin()) {
            require_once FLOATING_PROMO_PLUGIN_PATH . 'includes/admin-settings.php';
            new FloatingPromoAdmin();
        }

        // Output the promo banner
        add_action('wp_footer', array($this, 'display_promo_banner'));
        
        // AJAX handler for image upload
        add_action('wp_ajax_floating_promo_upload_image', array($this, 'handle_image_upload'));
    }

    public function enqueue_scripts() {
    // Enqueue media uploader in admin
        if (is_admin()) {
        // We'll handle admin scripts in the admin class
            return;
        }
        
    // Enqueue CSS
        wp_enqueue_style(
            'floating-promo-style',
            FLOATING_PROMO_PLUGIN_URL . 'css/floating-promo.css',
            array(),
            FLOATING_PROMO_VERSION
        );

    // Enqueue JavaScript
        wp_enqueue_script(
            'floating-promo-script',
            FLOATING_PROMO_PLUGIN_URL . 'js/floating-promo.js',
            array('jquery'),
            FLOATING_PROMO_VERSION,
            true
        );

    // Localize script for AJAX and translations
        wp_localize_script('floating-promo-script', 'floating_promo_params', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('floating_promo_nonce')
        ));
    }
    
    public function handle_image_upload() {
        // Check nonce for security
        if (!wp_verify_nonce($_POST['nonce'], 'floating_promo_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check user capabilities
        if (!current_user_can('upload_files')) {
            wp_die('You do not have permission to upload files');
        }
        
        // Upload file
        $file = $_FILES['floating_promo_image'];
        $status = wp_handle_upload($file, array('test_form' => false));
        
        if (isset($status['error'])) {
            wp_send_json_error($status['error']);
        }
        
        $url = $status['url'];
        wp_send_json_success($url);
    }

    public function display_promo_banner() {
    // Get plugin options
        $options = get_option('floating_promo_settings');
        
    // Check if banner is enabled
        if (empty($options['enable_banner']) || !$options['enable_banner']) {
            return;
        }
        
    // Check if current page is excluded
        $excluded_pages = isset($options['excluded_pages']) ? $options['excluded_pages'] : array();
        if (is_page($excluded_pages) || is_single($excluded_pages)) {
            return;
        }
        
    // Get banner content
        $banner_content = isset($options['banner_content']) ? $options['banner_content'] : '';
        $banner_url = isset($options['banner_url']) ? $options['banner_url'] : '#';
        $banner_image = isset($options['banner_image']) ? $options['banner_image'] : '';
        $banner_position = isset($options['banner_position']) ? $options['banner_position'] : 'center';
        
        if (empty($banner_content) && empty($banner_image)) {
            return;
        }
        
    // Get saved position from cookie or use default
        $position = isset($_COOKIE['floating_promo_position']) ? $_COOKIE['floating_promo_position'] : $banner_position;
        $is_closed = isset($_COOKIE['floating_promo_closed']) ? $_COOKIE['floating_promo_closed'] : 'false';
        
    // Don't display if closed
        if ($is_closed === 'true') {
            return;
        }
        
    // Output the banner
        ?>
        <div id="floating-promo-banner" class="floating-promo-<?php echo esc_attr($position); ?>">
            <div class="floating-promo-content">
                <a href="<?php echo esc_url($banner_url); ?>" target="_blank" class="floating-promo-link">
                    <?php if (!empty($banner_image)) : ?>
                        <div class="floating-promo-image">
                            <img src="<?php echo esc_url($banner_image); ?>" alt="<?php esc_attr_e('Promo Banner', 'floating-promo'); ?>">
                        </div>
                    <?php endif; ?>
                    <?php if (!empty($banner_content)) : ?>
                        <div class="floating-promo-text">
                            <?php echo wp_kses_post($banner_content); ?>
                        </div>
                    <?php endif; ?>
                </a>
                <div class="floating-promo-controls">
                    <button class="floating-promo-move" title="<?php esc_attr_e('Pindah Posisi', 'floating-promo'); ?>">
                        <span class="dashicons dashicons-move"></span>
                    </button>
                    <button class="floating-promo-close" title="<?php esc_attr_e('Tutup', 'floating-promo'); ?>">
                        <span class="dashicons dashicons-no"></span>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }
}

// Initialize the plugin
FloatingPromo::get_instance();

// Activation hook
register_activation_hook(__FILE__, 'floating_promo_activate');
function floating_promo_activate() {
    // Set default options
    $default_options = array(
        'enable_banner' => true,
        'banner_content' => '<h3>Promo Spesial!</h3><p>Dapatkan diskon 20% untuk semua paket perjalanan</p>',
        'banner_url' => '#',
        'banner_image' => FLOATING_PROMO_PLUGIN_URL . 'images/default-banner.jpg',
        'banner_position' => 'center',
        'excluded_pages' => array()
    );
    
    if (!get_option('floating_promo_settings')) {
        add_option('floating_promo_settings', $default_options);
    }
}

// Deactivation hook
register_deactivation_hook(__FILE__, 'floating_promo_deactivate');
function floating_promo_deactivate() {
    // Clean up if needed
}
?>